//===============================================================================================
// Object:			zFormsetList and zFormset
// Description:	Responsible for help bucket information associated with each formset.
//
// Copyright <c> 2003 Intuit, Inc.  All rights reserved
//===============================================================================================

var kApplicationFormsetId = "APPLICATION";

//-----------------------------------------------------------------------------------------------
function getXmlBuckets( xml ){ return xml.selectNodes( "/helpConfig/buckets//bucket" ); }
	// Retrieves one or more bucket hierarchies from the xml parser

//-----------------------------------------------------------------------------------------------
// zBucket - encapsulates the properties of a bucket
//-----------------------------------------------------------------------------------------------
function zBucket( node, basePath )
	// Initialize from the associated xml bucket node
{
	this.baseRef = basePath + '/';
	
	var child = null;
	var cnt = node.childNodes.length;
	for( var i=0; i<cnt; ++i )
	{
		child = node.childNodes[i];
		
		switch( child.nodeName )
		{
			case "type":			this.type = child.text.toLowerCase(); break;
			case "typename":		this.typename = child.text.toLowerCase(); break;
			case "description":	this.description = child.text; break;
			case "index":			this.index = child.text; break;
			case "version":		this.version = child.text; break;

			case "combinerHash":
				{
					var hashNum = parseInt( child.text );
					if( isNaN( hashNum ) )
					{
						debugMsg( "combinerHash must be a num", true );
					}	
					else
					{
						this.combined = true;
						this.combinerHash = hashNum;
					}
				}	
				break;
				
			case "variant_set":
				{				
					var vCnt = child.childNodes.length;
					if( vCnt>0 )
					{
						this.variants = new Array( vCnt );
						
						for( var vi=0; vi<vCnt; ++vi )
						{
							if( "variant" == child.childNodes[vi].nodeName )
							{
								this.variants[vi] = child.childNodes[vi].text.toLowerCase();
							}	
						}
					}	
				}
				break;

			default: break; // ignore				
		}
	}
}

//-----------------------------------------------------------------------------------------------
function _zBucketIsEnabled( prodVariant )
	// returns true if the bucket is enabled for the specified variant
{
	if( !this.variants ) return true; // no specified variants...interpret as all
	
	var cnt = this.variants.length;
	if( 1 == cnt && "all" == this.variants[0] ) return true; 
	
	var name = prodVariant.toLowerCase();
	
	for( var i=0; i<cnt; ++i ) if( name==this.variants[i] ) return true;

	return false;
}
		
//-----------------------------------------------------------------------------------------------
p = zBucket.prototype;
p.version = "";
p.baseRef = "";
p.type = "";
p.typename = "";
p.description = "";
p.index = "";
p.combined = false;
p.combinerHash = null;
p.variants = null;

p.isEnabled = _zBucketIsEnabled;

// private
p = null;


//-----------------------------------------------------------------------------------------------
// zConfig - encapsulates the configuration information obtains from the config file
//-----------------------------------------------------------------------------------------------
function zConfig( formset )
{
	var cfgPath = formset.path;
	cfgPath += '/' + formset.id + '.config';
	
	if( gbDebug ) debugMsg( "loading config file: " + cfgPath );
	
	var xml = getXMLParser();
	if( !loadXml( xml, cfgPath ) ) return;
	
	this._initBuckets( xml, formset.path );
	
	xml = null;	
}

//-----------------------------------------------------------------------------------------------
function _zConfigInitBuckets( xml, baseRef )
{
	var nodes = getXmlBuckets( xml );
	
	var cnt = nodes.length;
	if( cnt > 0 ) 
	{	
		this.buckets = new Array( cnt );		
		for( var i=0; i<cnt; ++i ) this.buckets[i] = new zBucket( nodes[i], baseRef );
	}
}

//-----------------------------------------------------------------------------------------------
p = zConfig.prototype;
p.buckets = null;

// private
p._initBuckets = _zConfigInitBuckets;
p = null;


//-----------------------------------------------------------------------------------------------
// zFormset - encapsulates help metadata information for a formset
//-----------------------------------------------------------------------------------------------
function zFormset( id, path )
{
	this.id = (id) ? id.toLowerCase() : null;	
	this.path = path;
	this.config = null;
}

//-----------------------------------------------------------------------------------------------
function _zFormsetLoad()
	// Load the configuration information associated with the formset
	// Don't put in ctor...zFormset also use to check if one is already cached
{
	this.config = new zConfig( this );
}

//-----------------------------------------------------------------------------------------------
function _zFormsetBuildSearchContext( variant )
	// Build a list of index paths that are relevant for the specifed variant
{
	if( !this.config || !this.config.buckets ) return "";
	
	var context = "";
	var bk, cnt =this.config.buckets.length;
	
	for( var i=0; i<cnt; ++i )
	{
		if( this.config.buckets[i].isEnabled( variant ) )
		{
			bk = this.config.buckets[i];
			
			if( bk.index.length>0 )
			{
				context += bk.baseRef;
				context += bk.index;
				context += '|';
			}
		}	
	}

	return context;
}

//-----------------------------------------------------------------------------------------------
function _zFormsetGetBucket( type )
	// Returns the bucket associated with the specified typename
{
	if( type && this.config && this.config.buckets )
	{
		var cnt = this.config.buckets.length;
		
		for( var i=0; i<cnt; ++i )
		{
			if( type == this.config.buckets[i].type ) return this.config.buckets[i];
		}
	}
	
	return null;	
}

//-----------------------------------------------------------------------------------------------
p = zFormset.prototype;
p.load = _zFormsetLoad;
p.buildSearchContext = _zFormsetBuildSearchContext;
p.getBucket = _zFormsetGetBucket;
p = null;


//-----------------------------------------------------------------------------------------------
// zFormsetList - hold a list of formsets
//-----------------------------------------------------------------------------------------------
function zFormsetList()
{
	this.formsets = new Array();	
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListFind( id )
	// Locates the formset with the associated id
{
	if(!id) return;
	
	var cnt = this.formsets.length;
	id = id.toLowerCase();
	
	for( var i=0; i<cnt; ++i )
	{
		if( id == this.formsets[i].id ) return this.formsets[i];
	}
	
	return null;
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListAdd( fs )
	// Take ownership of the specified formset and add it to the list
{
	if( !fs ) return;
	if( this.find( fs.id ) ) return;	// already exists...do not replace

	// Append it to the list
	this.formsets.push( fs );	
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListAddList( list )
	// Add the array of formsets into this list if not already present
{
	if( !list ) return;
	
	var fs, cfs;
	var cnt = list.length;
	
	// Add each formset
	for( var i=0; i<cnt; ++i )
	{
		fs = list[i];

		// check if the formset is already in list of cached formsets
		cfs = this.find( fs.id );
		if( !cfs )
		{
			// Not cached...load in the configuration associated with the formset
			// and cache it.
			fs.load();
			this.add( fs );
		}		
	}
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListBuildContext( formsetCache )
	// Populate this list based on the current engine formset.
	// Use the formset cache to retrieve and cache the formset info
{
	if( !formsetCache ) return;
	
	// Ask the C++ handler for the list of active formsets
	onCmd( "initSearchContext" );

	// Get the result sent back from the C++ handler
	var args = getLastCmdResultList();
	if( !args || args.length < 2 ) return;

	// Add the specified list of formset to the formset list cache
	this.productVariant = args[0];
	var formsets = args[1];			

	// Read the information about each formset from teh config file and cache it
	formsetCache.addList( formsets );
			
	// Create a list of the formsets to be searched
	var cnt = formsets.length;
	for( var i=0; i<cnt; ++i ) this.add( formsetCache.find( formsets[i].id ) );				
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListBuildSearchContext()
	// Build a list of index paths that are relevant for the specifed variant
{
	var context = "";
	var cnt = this.formsets.length;
	
	for( var i=0; i<cnt; ++i )
	{
		context += this.formsets[i].buildSearchContext( this.productVariant );
	}
	return context;
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListGetBucket( type )
	// Returns the first matching bucket for the specified type
{
	if( !type ) return null;
	
	var bucket = null;
	var cnt = this.formsets.length;
	
	// App formset is first, give precedence to tax formsets (temp workaround)
	for( var i=cnt-1; i>=0; --i )
	{
		bucket = this.formsets[i].getBucket( type );
		if( bucket ) break;
	}
	
	return bucket;
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListGetDescription( type )
	// Returns the description for the associated type
{
	var bucket = this.getBucket( type );
	return (bucket) ? bucket.description : "";
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListGetPath( type )
	// Returns the base path for the associated type
{
	var bucket = this.getBucket( type );
	return (bucket) ? bucket.baseRef : "";
}

//-----------------------------------------------------------------------------------------------
function _zFormsetListGetSynonymFilename()
{
	var fs = this.find( kApplicationFormsetId );
	if( !fs ) return null;
	
	return fs.path + '/' + 'synonyms.rul';
}

//-----------------------------------------------------------------------------------------------
p = zFormsetList.prototype;
p.find = _zFormsetListFind;
p.add = _zFormsetListAdd;
p.addList = _zFormsetListAddList;
p.buildContext = _zFormsetListBuildContext;
p.buildSearchContext = _zFormsetListBuildSearchContext;
p.getBucket = _zFormsetListGetBucket;
p.getDescription = _zFormsetListGetDescription;
p.getPath = _zFormsetListGetPath;
p.getSynonymFilename = _zFormsetListGetSynonymFilename

p.productVariant = null;

p = null;
